import 'package:dartz/dartz.dart';
import 'package:razinsoft_rider/data/models/forgot_verify_otp_response/forgot_verify_otp_response.dart';
import 'package:razinsoft_rider/data/models/request_otp_response/request_otp_response.dart';
import 'package:razinsoft_rider/data/models/reset_password_response.dart';
import 'package:razinsoft_rider/data/repositories/base_repository.dart';
import 'package:razinsoft_rider/domain/interfaces/auth_service_interface.dart';

import '../../core/errors/failure.dart';
import '../models/common_response.dart';
import '../models/login_response.dart';
import '../models/login_with_password_response.dart';
import '../models/logout_response.dart';
import '../models/otp_verify_response.dart';
import '../models/profile_update_response.dart';
import '../models/resend_otp_model/resend_otp_mode.dart';
import '../models/rider_details_response/rider_details_response.dart';
import 'interfaces/auth_repo_interface.dart';

class AuthRepoImpl extends BaseRepository implements IAuthRepo {
  final IAuthService _authService;

  AuthRepoImpl({required IAuthService authService})
      : _authService = authService;

  @override
  Future<Either<Failure, LoginResponse>> login({required String phone, required String countryCode, String? deviceToken}) async => await safeApiCall(() async {
      final response = await _authService.login(phone: phone, countryCode: countryCode, deviceToken: deviceToken);
      return LoginResponse.fromJson(response.data);
    });

  @override
  Future<Either<Failure, LoginWithPasswordResponse>> resendSignIn({required num? userId, required String? deviceToken}) async => await safeApiCall(() async {
    final response = await _authService.resendSignIn(userId: userId, deviceToken: deviceToken);
    return LoginWithPasswordResponse.fromJson(response.data);
  });

  @override
  Future<Either<Failure, LoginWithPasswordResponse>> loginWithPassword(
      {required String mobile, required String password, String? deviceToken, String? wantLogin}) async => await safeApiCall(() async {
      final response =
          await _authService.loginWithPassword(mobile: mobile, password: password, deviceToken: deviceToken, wantLogin: wantLogin);
      try{
        return LoginWithPasswordResponse.fromJson(response.data);
      }catch(e){
        throw Failure(message: 'Failed to parse login response: $e');      }
    });

  @override
  Future<Either<Failure, OtpVerifyResponse>> verifyOtp(
      {required String mobile, required String otp, String? deviceToken, String? wantLogin}) => safeApiCall(() async {
      final response = await _authService.verifyOtp(mobile: mobile, otp: otp, deviceToken: deviceToken, wantLogin: wantLogin);
      return OtpVerifyResponse.fromJson(response.data);
    });

  @override
  Future<Either<Failure, CommonResponse>> updatePassword(
      {required String password}) async => safeApiCall(() async {
      final response = await _authService.updatePassword(password: password);
      return CommonResponse.fromMap(response.data);
    });

  @override
  Future<Either<Failure, CommonResponse>> changePassword({required String currentPassword, required String newPassword, required newConfirmPassword}) async => await safeApiCall(() async {
      final response = await _authService.changePassword(currentPassword: currentPassword, newPassword: newPassword, newConfirmPassword: newConfirmPassword);
      return CommonResponse.fromMap(response.data);
    });

  @override
  Future<Either<Failure, ProfileUpdateResponse>> profileUpdate(
      {required Map<String, dynamic> data}) async => safeApiCall(() async {
      final response = await _authService.updateProfile(data: data);
      return ProfileUpdateResponse.fromJson(response.data);
    });

  @override
  Future<Either<Failure, LogoutResponse>> logout() async => safeApiCall(() async {
      final response = await _authService.logout();
      return LogoutResponse.fromMap(response.data);
    });

  @override
  Future<Either<Failure, ResendOtpModel>> resendOTP(
      {required String mobile}) async => safeApiCall(() async {
      final response = await _authService.resendOTP(mobile: mobile);
      return ResendOtpModel.fromJson(response.data);
    });

  @override
  Future<Either<Failure, CommonResponse>> forgotPassword(
      {required String mobile}) async => safeApiCall(() async {
      final response = await _authService.forgotPassword(mobile: mobile);
      return CommonResponse.fromMap(response.data);
    });

  @override
  Future<Either<Failure, CommonResponse>> updateProfilePhoto(
      {required String imagePath}) async => safeApiCall(() async {
      final response =
          await _authService.updateProfilePhoto(imagePath: imagePath);
      return CommonResponse.fromMap(response.data);
    });

  @override
  Future<Either<Failure, RiderDetailsResponse>> riderDetails() async => safeApiCall(() async {
      final response = await _authService.riderDetails();
      return RiderDetailsResponse.fromMap(response.data);
    });

  @override
  Future<Either<Failure, RequestOtpResponse>> requestOTP(
      {required String mobile}) async => safeApiCall(() async {
      final response = await _authService.requestOTP(mobile: mobile);
      return RequestOtpResponse.fromMap(response.data);
    });

  @override
  Future<Either<Failure, ForgotVerifyOtpResponse>> forgetVerifyOtp(
      {required String mobile, required String otp}) async => safeApiCall(() async {
      final response =
          await _authService.forgetVerifyOtp(mobile: mobile, otp: otp);
      return ForgotVerifyOtpResponse.fromMap(response.data);
    });

  @override
  Future<Either<Failure, ResetPasswordResponse>> resetPassword(
      {required Map<String, dynamic> data}) async => safeApiCall(() async {
      final response = await _authService.resetPassword(data: data);
      return ResetPasswordResponse.fromMap(response.data);
    });
}
